<?php

namespace Tests\Feature;

use App\Models\Setting;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Mail;
use Tests\TestCase;

class EmailSettingsTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();

        // Create an admin user for testing
        $this->admin = User::factory()->create([
            'role' => 'admin',
        ]);
    }

    public function test_test_email_endpoint_validates_required_fields(): void
    {
        $response = $this->actingAs($this->admin)->postJson(route('admin.settings.email.test'), [
            // Missing required fields
        ]);

        $response->assertStatus(422);
        $response->assertJsonValidationErrors([
            'mail_mailer',
            'mail_host',
            'mail_port',
            'mail_from_name',
            'mail_from_address',
        ]);
    }

    public function test_test_email_endpoint_returns_success_with_valid_data(): void
    {
        Mail::fake();

        $response = $this->actingAs($this->admin)->postJson(route('admin.settings.email.test'), [
            'mail_mailer' => 'log', // Use log mailer to avoid SMTP connection
            'mail_host' => 'smtp.example.com',
            'mail_port' => 587,
            'mail_encryption' => 'tls',
            'mail_username' => 'test@example.com',
            'mail_password' => 'password',
            'mail_from_name' => 'Test Mailer',
            'mail_from_address' => 'test@example.com',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
    }

    public function test_mail_config_provider_uses_smtp_username_as_from_address(): void
    {
        $smtpUsername = 'test@example.com';
        $differentFromAddress = 'noreply@example.com';

        Setting::set('mail_mailer', 'smtp');
        Setting::set('mail_username', $smtpUsername);
        Setting::set('mail_from_address', $differentFromAddress);

        // Re-register the mail config service provider to load the settings
        $provider = new \App\Providers\MailConfigServiceProvider($this->app);
        $provider->boot();

        // Check that the from address is set to the SMTP username
        $this->assertEquals($smtpUsername, config('mail.from.address'));
    }

    public function test_mail_config_uses_from_address_when_smtp_username_is_empty(): void
    {
        $fromAddress = 'noreply@example.com';

        Setting::set('mail_mailer', 'smtp');
        Setting::set('mail_username', '');
        Setting::set('mail_from_address', $fromAddress);

        // Re-register the mail config service provider to load the settings
        $provider = new \App\Providers\MailConfigServiceProvider($this->app);
        $provider->boot();

        // Check that the from address is used when no SMTP username is provided
        $this->assertEquals($fromAddress, config('mail.from.address'));
    }

    public function test_test_email_sends_to_site_email_by_default(): void
    {
        Mail::fake();

        $smtpUsername = 'test@example.com';

        $response = $this->actingAs($this->admin)->postJson(route('admin.settings.email.test'), [
            'mail_mailer' => 'log',
            'mail_host' => 'smtp.example.com',
            'mail_port' => 587,
            'mail_encryption' => 'tls',
            'mail_username' => $smtpUsername,
            'mail_password' => 'password',
            'mail_from_name' => 'Test Mailer',
            'mail_from_address' => 'test@example.com',
        ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'message' => 'Test email sent successfully to '.$smtpUsername,
        ]);
    }

    public function test_test_email_sends_to_custom_recipient_when_specified(): void
    {
        Mail::fake();

        $customRecipient = 'admin@example.com';
        $smtpUsername = 'test@example.com';

        $response = $this->actingAs($this->admin)->postJson(route('admin.settings.email.test'), [
            'mail_mailer' => 'log',
            'mail_host' => 'smtp.example.com',
            'mail_port' => 587,
            'mail_encryption' => 'tls',
            'mail_username' => $smtpUsername,
            'mail_password' => 'password',
            'mail_from_name' => 'Test Mailer',
            'mail_from_address' => 'test@example.com',
            'test_email_recipient' => $customRecipient,
        ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'message' => 'Test email sent successfully to '.$customRecipient,
        ]);
    }

    public function test_test_email_uses_from_address_when_no_smtp_username(): void
    {
        Mail::fake();

        $fromAddress = 'test@example.com';

        $response = $this->actingAs($this->admin)->postJson(route('admin.settings.email.test'), [
            'mail_mailer' => 'log',
            'mail_host' => 'smtp.example.com',
            'mail_port' => 587,
            'mail_encryption' => 'tls',
            'mail_username' => '',
            'mail_password' => 'password',
            'mail_from_name' => 'Test Mailer',
            'mail_from_address' => $fromAddress,
        ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'message' => 'Test email sent successfully to '.$fromAddress,
        ]);
    }

    public function test_email_settings_update_saves_test_recipient(): void
    {
        $testRecipient = 'admin@example.com';

        $this->actingAs($this->admin)->post(route('admin.settings.email.update'), [
            'mail_mailer' => 'smtp',
            'mail_host' => 'smtp.example.com',
            'mail_port' => 587,
            'mail_encryption' => 'tls',
            'mail_username' => 'test@example.com',
            'mail_password' => 'password',
            'mail_from_name' => 'Test Mailer',
            'mail_from_address' => 'test@example.com',
            'test_email_recipient' => $testRecipient,
        ]);

        // Verify the test recipient was saved
        $this->assertEquals($testRecipient, Setting::get('test_email_recipient'));
    }
}
