@extends('layouts.app')
@section('title', 'Domain Search')
@section('content')
<div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
    <div class="text-center mb-12">
        <h1 class="text-4xl font-bold text-white mb-4">Find Your Perfect Domain</h1>
        <p class="text-gray-400 text-lg">Search for available domain names and register them instantly</p>
    </div>

    <div class="bg-gray-800 rounded-lg border border-gray-700 p-8 mb-8" x-data="domainSearch()">
        <div class="flex gap-4">
            <input type="text" x-model="domain" @keyup.enter="search()"
                placeholder="Enter your domain name (e.g., mywebsite.com)"
                class="flex-1 px-6 py-4 bg-gray-900 border border-gray-700 rounded-lg text-white text-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
            <button @click="search()" :disabled="searching"
                class="px-8 py-4 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white font-semibold rounded-lg transition shadow-lg disabled:opacity-50"
                :class="{'opacity-50 cursor-not-allowed': searching}">
                <span x-show="!searching">Search</span>
                <span x-show="searching">Searching...</span>
            </button>
        </div>

        <div x-show="result" class="mt-6">
            <div x-show="result && result.success && result.available" 
                class="bg-green-900 border border-green-700 rounded-lg p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <h3 class="text-xl font-bold text-green-300 mb-2">
                            <svg class="inline w-6 h-6 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                            </svg>
                            <span x-text="result.domain"></span> is available!
                        </h3>
                        <div class="text-green-200 space-x-4">
                            <span>Register: <strong x-text="formatCurrency(result.pricing.register)"></strong></span>
                            <span>Renew: <strong x-text="formatCurrency(result.pricing.renew)"></strong></span>
                        </div>
                    </div>
                    <form action="{{ route('cart.add.domain') }}" method="POST">
                        @csrf
                        <input type="hidden" name="domain" :value="result.domain.split('.')[0]">
                        <input type="hidden" name="extension" :value="'.' + result.domain.split('.').slice(1).join('.')">
                        <input type="hidden" name="action" value="register">
                        <button type="submit" class="px-6 py-3 bg-white text-gray-900 font-semibold rounded-lg hover:bg-gray-100 transition">
                            Add to Cart
                        </button>
                    </form>
                </div>
            </div>

            <div x-show="result && result.success && !result.available"
                class="bg-red-900 border border-red-700 rounded-lg p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <h3 class="text-xl font-bold text-red-300 mb-2">
                            <svg class="inline w-6 h-6 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                            </svg>
                            <span x-text="result.domain"></span> is not available
                        </h3>
                        <p class="text-red-200">This domain is already registered. Try a different name or extension.</p>
                    </div>
                    <form action="{{ route('cart.add.domain') }}" method="POST">
                        @csrf
                        <input type="hidden" name="domain" :value="result.domain.split('.')[0]">
                        <input type="hidden" name="extension" :value="'.' + result.domain.split('.').slice(1).join('.')">
                        <input type="hidden" name="action" value="transfer">
                        <button type="submit" class="px-6 py-3 bg-white text-gray-900 font-semibold rounded-lg hover:bg-gray-100 transition">
                            Transfer
                        </button>
                    </form>
                </div>
            </div>

            <div x-show="result && !result.success"
                class="bg-yellow-900 border border-yellow-700 rounded-lg p-4">
                <p class="text-yellow-200" x-text="result.message"></p>
            </div>
        </div>
    </div>

    <div class="bg-gray-800 rounded-lg border border-gray-700 p-8">
        <h2 class="text-2xl font-bold text-white mb-6">Popular Domain Extensions</h2>
        <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
            @php
            $popular = ['.com', '.net', '.org', '.io', '.co', '.app', '.dev', '.xyz'];
            @endphp
            @foreach($popular as $ext)
                @php
                $pricing = \App\Models\DomainPricing::where('extension', $ext)->where('active', true)->first();
                @endphp
                @if($pricing)
                <div class="bg-gray-900 border border-gray-700 rounded-lg p-4 text-center hover:border-blue-600 transition cursor-pointer" @click="domain += '{{ $ext }}'">
                    <div class="text-2xl font-bold text-blue-400 mb-2">{{ $ext }}</div>
                    <div class="text-white font-semibold">{{ format_currency($pricing->register_price) }}</div>
                    <div class="text-xs text-gray-400">per year</div>
                </div>
                @endif
            @endforeach
        </div>
    </div>
</div>

<script>
function domainSearch() {
    return {
        domain: '',
        searching: false,
        result: null,
        async search() {
            if (!this.domain) return;
            
            this.searching = true;
            this.result = null;
            
            try {
                const response = await fetch('{{ route('domain.check') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({ domain: this.domain })
                });
                
                this.result = await response.json();
            } catch (error) {
                this.result = {
                    success: false,
                    message: 'Error checking domain availability'
                };
            } finally {
                this.searching = false;
            }
        },
        formatCurrency(amount) {
            return '$' + parseFloat(amount).toFixed(2);
        }
    }
}
</script>
@endsection
