@extends('layouts.app')

@section('title', 'Server Settings')

@section('content')
<div
    class="min-h-screen bg-gray-900 p-6"
    x-data="serverManagement()"
>
    <div class="max-w-7xl mx-auto">
        <!-- Breadcrumb -->
        <nav class="flex mb-6" aria-label="Breadcrumb">
            <ol class="inline-flex items-center space-x-1 md:space-x-3">
                <li class="inline-flex items-center">
                    <a href="{{ route('admin.dashboard') }}" class="text-gray-400 hover:text-white">Admin</a>
                </li>
                <li>
                    <div class="flex items-center">
                        <svg class="w-6 h-6 text-gray-400" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clip-rule="evenodd"></path>
                        </svg>
                        <a href="{{ route('admin.settings.index') }}" class="ml-1 text-gray-400 hover:text-white">Settings</a>
                    </div>
                </li>
                <li aria-current="page">
                    <div class="flex items-center">
                        <svg class="w-6 h-6 text-gray-400" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clip-rule="evenodd"></path>
                        </svg>
                        <span class="ml-1 text-gray-500">Server Settings</span>
                    </div>
                </li>
            </ol>
        </nav>

        <!-- Header -->
        <div class="mb-8 flex justify-between items-center">
            <div>
                <h1 class="text-3xl font-bold text-white mb-2">Server Management</h1>
                <p class="text-gray-400">Configure and manage hosting servers for provisioning</p>
            </div>
            <button @click="showAddForm = true" 
                    class="px-6 py-3 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white font-semibold rounded-lg transition shadow-lg flex items-center">
                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                </svg>
                Add New Server
            </button>
        </div>

        @if(session('success'))
            <div class="mb-6 p-4 bg-green-500/10 border border-green-500 text-green-400 rounded-lg flex items-center">
                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                </svg>
                {{ session('success') }}
            </div>
        @endif

        <!-- Add Server Form Modal -->
        <div x-show="showAddForm" 
             x-transition
             class="fixed inset-0 bg-black/50 backdrop-blur-sm z-50 flex items-center justify-center p-4"
             @click.self="showAddForm = false">
            <div class="bg-gray-800 border border-gray-700 rounded-xl shadow-2xl max-w-4xl w-full max-h-[90vh] overflow-y-auto">
                <div class="sticky top-0 bg-gray-800 border-b border-gray-700 px-6 py-4 flex justify-between items-center">
                    <h2 class="text-xl font-bold text-white">Add New Server</h2>
                    <button @click="showAddForm = false" class="text-gray-400 hover:text-white">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>

                <form action="{{ route('admin.settings.servers.store') }}" method="POST" class="p-6">
                    @csrf

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Server Name -->
                        <div>
                            <label class="block text-sm font-medium text-gray-300 mb-2">Server Name *</label>
                            <input type="text" name="name" value="{{ old('name') }}" required
                                class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="e.g., Server 1">
                            @error('name')<p class="mt-1 text-sm text-red-400">{{ $message }}</p>@enderror
                        </div>

                        <!-- Server Type -->
                        <div>
                            <label class="block text-sm font-medium text-gray-300 mb-2">Server Type *</label>
                            <select name="type" required x-model="serverType"
                                class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="">Select Type</option>
                                <option value="cpanel">cPanel/WHM</option>
                                <option value="directadmin">DirectAdmin</option>
                                <option value="plesk">Plesk</option>
                                <option value="custom">Custom</option>
                            </select>
                            @error('type')<p class="mt-1 text-sm text-red-400">{{ $message }}</p>@enderror
                        </div>

                        <!-- Hostname -->
                        <div>
                            <label class="block text-sm font-medium text-gray-300 mb-2">Hostname *</label>
                            <input type="text" name="hostname" value="{{ old('hostname') }}" required
                                x-model="hostname"
                                class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="server1.example.com">
                            @error('hostname')<p class="mt-1 text-sm text-red-400">{{ $message }}</p>@enderror
                        </div>

                        <!-- IP Address -->
                        <div>
                            <label class="block text-sm font-medium text-gray-300 mb-2">IP Address *</label>
                            <input type="text" name="ip_address" value="{{ old('ip_address') }}" required
                                class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="192.168.1.1">
                            @error('ip_address')<p class="mt-1 text-sm text-red-400">{{ $message }}</p>@enderror
                        </div>

                        <!-- Username -->
                        <div>
                            <label class="block text-sm font-medium text-gray-300 mb-2">Username *</label>
                            <input type="text" name="username" value="{{ old('username') }}" required
                                x-model="username"
                                class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="root">
                            @error('username')<p class="mt-1 text-sm text-red-400">{{ $message }}</p>@enderror
                        </div>

                        <!-- Password -->
                        <div>
                            <label class="block text-sm font-medium text-gray-300 mb-2">Password *</label>
                            <div class="relative">
                                <input :type="showPassword ? 'text' : 'password'" name="password" required
                                    x-model="password"
                                    class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500 pr-10">
                                <button type="button" @click="showPassword = !showPassword" 
                                    class="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 hover:text-white">
                                    <svg x-show="!showPassword" class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path>
                                    </svg>
                                    <svg x-show="showPassword" class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.88 9.88l-3.29-3.29m7.532 7.532l3.29 3.29M3 3l3.59 3.59m0 0A9.953 9.953 0 0112 5c4.478 0 8.268 2.943 9.543 7a10.025 10.025 0 01-4.132 5.411m0 0L21 21"></path>
                                    </svg>
                                </button>
                            </div>
                            @error('password')<p class="mt-1 text-sm text-red-400">{{ $message }}</p>@enderror
                        </div>

                        <!-- Port -->
                        <div>
                            <label class="block text-sm font-medium text-gray-300 mb-2">Port *</label>
                            <input type="number" name="port" value="{{ old('port', 2087) }}" required
                                x-model="port"
                                class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500">
                            @error('port')<p class="mt-1 text-sm text-red-400">{{ $message }}</p>@enderror
                            <p class="mt-1 text-xs text-gray-400">cPanel/WHM: 2087, DirectAdmin: 2222, Plesk: 8443</p>
                        </div>

                        <!-- Max Accounts -->
                        <div>
                            <label class="block text-sm font-medium text-gray-300 mb-2">Max Accounts</label>
                            <input type="number" name="max_accounts" value="{{ old('max_accounts', 0) }}" min="0"
                                class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                                placeholder="0 for unlimited">
                            @error('max_accounts')<p class="mt-1 text-sm text-red-400">{{ $message }}</p>@enderror
                        </div>
                    </div>

                    <!-- Nameservers -->
                    <div class="mt-6">
                        <h3 class="text-lg font-semibold text-white mb-4">Nameservers</h3>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-2">Nameserver 1</label>
                                <input type="text" name="nameserver1" value="{{ old('nameserver1') }}"
                                    class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                                    placeholder="ns1.example.com">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-2">Nameserver 2</label>
                                <input type="text" name="nameserver2" value="{{ old('nameserver2') }}"
                                    class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                                    placeholder="ns2.example.com">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-2">Nameserver 3</label>
                                <input type="text" name="nameserver3" value="{{ old('nameserver3') }}"
                                    class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                                    placeholder="ns3.example.com">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-2">Nameserver 4</label>
                                <input type="text" name="nameserver4" value="{{ old('nameserver4') }}"
                                    class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                                    placeholder="ns4.example.com">
                            </div>
                        </div>
                    </div>

                    <!-- Notes -->
                    <div class="mt-6">
                        <label class="block text-sm font-medium text-gray-300 mb-2">Notes</label>
                        <textarea name="notes" rows="3"
                            class="w-full px-4 py-2 bg-gray-900 border border-gray-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
                            placeholder="Additional notes about this server...">{{ old('notes') }}</textarea>
                    </div>

                    <!-- Options -->
                    <div class="mt-6 flex items-center space-x-6">
                        <label class="flex items-center">
                            <input type="checkbox" name="secure" value="1" checked
                                class="w-4 h-4 bg-gray-900 border-gray-700 rounded text-blue-600 focus:ring-blue-500 focus:ring-2">
                            <span class="ml-2 text-sm text-gray-300">Use HTTPS</span>
                        </label>

                        <label class="flex items-center">
                            <input type="checkbox" name="active" value="1" checked
                                class="w-4 h-4 bg-gray-900 border-gray-700 rounded text-blue-600 focus:ring-blue-500 focus:ring-2">
                            <span class="ml-2 text-sm text-gray-300">Active</span>
                        </label>
                    </div>

                    <!-- Test Connection Result -->
                    <div x-show="testResult.show" 
                         :class="testResult.success ? 'bg-green-500/10 border-green-500 text-green-400' : 'bg-red-500/10 border-red-500 text-red-400'"
                         class="mt-6 p-4 border rounded-lg">
                        <div class="flex items-center">
                            <svg x-show="testResult.success" class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                            </svg>
                            <svg x-show="!testResult.success" class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                            </svg>
                            <span x-text="testResult.message"></span>
                        </div>
                    </div>

                    <!-- Action Buttons -->
                    <div class="mt-8 flex items-center justify-between">
                        <button type="button" @click="testConnection()" :disabled="testing"
                            class="px-6 py-3 bg-yellow-600 hover:bg-yellow-700 text-white font-semibold rounded-lg transition flex items-center disabled:opacity-50 disabled:cursor-not-allowed">
                            <svg x-show="!testing" class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"></path>
                            </svg>
                            <svg x-show="testing" class="animate-spin w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                            </svg>
                            <span x-text="testing ? 'Testing...' : 'Test Connection'"></span>
                        </button>

                        <div class="flex space-x-4">
                            <button type="button" @click="showAddForm = false" 
                                class="px-6 py-3 bg-gray-700 hover:bg-gray-600 text-white font-semibold rounded-lg transition">
                                Cancel
                            </button>
                            <button type="submit"
                                class="px-6 py-3 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white font-semibold rounded-lg transition shadow-lg flex items-center">
                                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                                </svg>
                                Add Server
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>

        <!-- Servers List -->
        <div class="bg-gray-800 border border-gray-700 rounded-xl overflow-hidden">
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-700">
                    <thead class="bg-gray-900">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Name</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Type</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Hostname</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">IP Address</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Accounts</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Status</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-300 uppercase tracking-wider">Actions</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-700">
                        @forelse($servers as $server)
                        <tr class="hover:bg-gray-700/50 transition">
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="text-sm font-medium text-white">{{ $server->name }}</div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <span class="px-2 py-1 text-xs font-semibold rounded-full 
                                    {{ $server->type === 'cpanel' ? 'bg-blue-500/10 text-blue-400 border border-blue-500/20' : '' }}
                                    {{ $server->type === 'directadmin' ? 'bg-purple-500/10 text-purple-400 border border-purple-500/20' : '' }}
                                    {{ $server->type === 'plesk' ? 'bg-green-500/10 text-green-400 border border-green-500/20' : '' }}
                                    {{ $server->type === 'custom' ? 'bg-gray-500/10 text-gray-400 border border-gray-500/20' : '' }}">
                                    {{ strtoupper($server->type) }}
                                </span>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">{{ $server->hostname }}</td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">{{ $server->ip_address }}</td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-300">
                                <span class="font-medium">{{ $server->accounts_count ?? 0 }}</span>{{ $server->max_accounts > 0 ? ' / ' . $server->max_accounts : '' }}
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <span class="px-2 py-1 text-xs font-semibold rounded-full {{ $server->active ? 'bg-green-500/10 text-green-400 border border-green-500/20' : 'bg-red-500/10 text-red-400 border border-red-500/20' }}">
                                    {{ $server->active ? 'Active' : 'Inactive' }}
                                </span>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                <div class="flex items-center space-x-3">
                                    <button @click="testServerConnection({{ $server->id }})" 
                                        class="text-yellow-400 hover:text-yellow-300 transition"
                                        title="Test Connection">
                                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"></path>
                                        </svg>
                                    </button>
                                    <a href="{{ route('admin.settings.servers.edit', $server) }}" 
                                       class="text-blue-400 hover:text-blue-300 transition">
                                        Edit
                                    </a>
                                    <form action="{{ route('admin.settings.servers.destroy', $server) }}" method="POST" 
                                          class="inline" onsubmit="return confirm('Are you sure you want to delete this server?')">
                                        @csrf
                                        @method('DELETE')
                                        <button type="submit" class="text-red-400 hover:text-red-300 transition">Delete</button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="7" class="px-6 py-12 text-center">
                                <div class="flex flex-col items-center justify-center">
                                    <svg class="w-16 h-16 text-gray-600 mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 12h14M5 12a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v4a2 2 0 01-2 2M5 12a2 2 0 00-2 2v4a2 2 0 002 2h14a2 2 0 002-2v-4a2 2 0 00-2-2m-2-4h.01M17 16h.01"></path>
                                    </svg>
                                    <h3 class="text-lg font-semibold text-gray-400 mb-2">No servers configured</h3>
                                    <p class="text-gray-500 mb-4">Add your first server to start provisioning accounts</p>
                                    <button @click="showAddForm = true" 
                                        class="px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-lg transition">
                                        Add Server
                                    </button>
                                </div>
                            </td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>

        @if($servers->hasPages())
            <div class="mt-6">
                {{ $servers->links() }}
            </div>
        @endif
    </div>
</div>

<script>
function serverManagement() {
    return {
        showAddForm: false,
        showPassword: false,
        testing: false,
        testResult: {
            show: false,
            success: false,
            message: ''
        },
        serverType: '',
        hostname: '',
        username: '',
        password: '',
        port: 2087,

        async testConnection() {
            if (!this.hostname || !this.username || !this.password || !this.port) {
                this.testResult = {
                    show: true,
                    success: false,
                    message: 'Please fill in hostname, username, password, and port to test connection'
                };
                return;
            }

            this.testing = true;
            this.testResult.show = false;

            try {
                const response = await fetch('{{ route('admin.settings.servers.test') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({
                        hostname: this.hostname,
                        username: this.username,
                        password: this.password,
                        port: this.port,
                        type: this.serverType,
                        secure: document.querySelector('input[name="secure"]').checked
                    })
                });

                const data = await response.json();
                
                this.testResult = {
                    show: true,
                    success: data.success,
                    message: data.message
                };
            } catch (error) {
                this.testResult = {
                    show: true,
                    success: false,
                    message: 'Connection test failed: ' + error.message
                };
            } finally {
                this.testing = false;
            }
        },

        async testServerConnection(serverId) {
            try {
                const response = await fetch(`/admin/settings/servers/${serverId}/test`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                });

                const data = await response.json();
                alert(data.message);
            } catch (error) {
                alert('Connection test failed: ' + error.message);
            }
        }
    }
}
</script>
@endsection
