<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\Server;
use App\Models\Product;
use App\Models\ProductPricing;
use App\Models\DomainPricing;
use App\Models\Setting;
use App\Models\Template;
use App\Models\EmailTemplate;
use App\Models\PaymentGateway;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    public function run(): void
    {
        // Create admin user
        User::create([
            'name' => 'Admin User',
            'email' => 'admin@billingcs.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        // Create demo client user
        User::create([
            'name' => 'Demo Client',
            'email' => 'client@billingcs.com',
            'password' => Hash::make('password'),
            'role' => 'client',
            'company' => 'Demo Company',
        ]);

        // Create Sample Server
        $server = Server::create([
            'name' => 'Server 1 - cPanel',
            'hostname' => 'server1.example.com',
            'ip_address' => '192.168.1.100',
            'type' => 'cpanel',
            'username' => 'root',
            'password' => encrypt('password123'),
            'port' => 2087,
            'secure' => true,
            'active' => true,
            'max_accounts' => 100,
            'accounts_count' => 0,
            'nameserver1' => 'ns1.example.com',
            'nameserver2' => 'ns2.example.com',
        ]);

        // Create Sample Products with Pricing
        $starter = Product::create([
            'name' => 'Starter Hosting',
            'description' => 'Perfect for small websites and blogs',
            'type' => 'hosting',
            'active' => true,
            'server_id' => $server->id,
            'features' => [
                'disk' => '10 GB',
                'bandwidth' => '100 GB',
                'databases' => '10',
                'email_accounts' => '25',
                'ssl' => 'Free',
            ],
        ]);

        ProductPricing::create(['product_id' => $starter->id, 'billing_cycle' => 'monthly', 'price' => 4.99, 'setup_fee' => 0]);
        ProductPricing::create(['product_id' => $starter->id, 'billing_cycle' => 'annually', 'price' => 49.99, 'setup_fee' => 0]);

        $professional = Product::create([
            'name' => 'Professional Hosting',
            'description' => 'Ideal for growing businesses',
            'type' => 'hosting',
            'active' => true,
            'server_id' => $server->id,
            'features' => [
                'disk' => '50 GB',
                'bandwidth' => 'Unlimited',
                'databases' => 'Unlimited',
                'email_accounts' => 'Unlimited',
                'ssl' => 'Free',
            ],
        ]);

        ProductPricing::create(['product_id' => $professional->id, 'billing_cycle' => 'monthly', 'price' => 9.99, 'setup_fee' => 0]);
        ProductPricing::create(['product_id' => $professional->id, 'billing_cycle' => 'annually', 'price' => 99.99, 'setup_fee' => 0]);

        $business = Product::create([
            'name' => 'Business Hosting',
            'description' => 'For high-traffic websites',
            'type' => 'hosting',
            'active' => true,
            'server_id' => $server->id,
            'features' => ['disk' => '100 GB', 'bandwidth' => 'Unlimited', 'databases' => 'Unlimited', 'email_accounts' => 'Unlimited', 'ssl' => 'Free', 'dedicated_ip' => 'Included'],
        ]);

        ProductPricing::create(['product_id' => $business->id, 'billing_cycle' => 'monthly', 'price' => 19.99, 'setup_fee' => 0]);
        ProductPricing::create(['product_id' => $business->id, 'billing_cycle' => 'annually', 'price' => 199.99, 'setup_fee' => 0]);

        // Create VPS Product
        $vps = Product::create([
            'name' => 'VPS Standard',
            'description' => 'Virtual Private Server with full root access',
            'type' => 'vps',
            'active' => true,
            'features' => ['cpu' => '2 Cores', 'ram' => '4 GB', 'disk' => '80 GB SSD', 'bandwidth' => '2 TB', 'root_access' => 'Yes'],
        ]);

        ProductPricing::create(['product_id' => $vps->id, 'billing_cycle' => 'monthly', 'price' => 29.99, 'setup_fee' => 10]);

        // Create Domain Pricing (including Nigerian TLDs)
        $domainExtensions = [
            ['.com', 9.99, 12.99, 9.99], ['.net', 11.99, 14.99, 11.99], ['.org', 11.99, 14.99, 11.99],
            ['.io', 39.99, 45.99, 39.99], ['.co', 29.99, 32.99, 29.99], ['.app', 19.99, 22.99, 19.99],
            ['.dev', 19.99, 22.99, 19.99], ['.xyz', 4.99, 12.99, 4.99],
            // Nigerian TLDs (prices in NGN)
            ['.ng', 5000.00, 5000.00, 5000.00], 
            ['.com.ng', 3000.00, 3000.00, 3000.00], 
            ['.org.ng', 3000.00, 3000.00, 3000.00],
            ['.net.ng', 3000.00, 3000.00, 3000.00],
            ['.edu.ng', 3500.00, 3500.00, 3500.00],
            ['.gov.ng', 3500.00, 3500.00, 3500.00],
        ];

        foreach ($domainExtensions as $domain) {
            DomainPricing::create([
                'extension' => $domain[0],
                'register_price' => $domain[1],
                'renew_price' => $domain[2],
                'transfer_price' => $domain[3],
                'active' => true,
            ]);
        }

        // Seed Payment Gateways with real API configurations
        $gateways = [
            [
                'name' => 'PayPal',
                'slug' => 'paypal',
                'display_name' => 'PayPal',
                'description' => 'Pay securely with PayPal',
                'is_active' => true,
                'is_test_mode' => true,
                'config' => ['client_id' => '', 'secret' => '', 'mode' => 'sandbox', 'webhook_id' => ''],
                'supported_currencies' => ['USD', 'EUR', 'GBP', 'CAD', 'AUD'],
                'transaction_fee_percentage' => 2.9,
                'transaction_fee_fixed' => 0.30,
                'display_order' => 1,
            ],
            [
                'name' => 'Stripe',
                'slug' => 'stripe',
                'display_name' => 'Credit/Debit Card (Stripe)',
                'description' => 'Pay with credit or debit card',
                'is_active' => true,
                'is_test_mode' => true,
                'config' => ['public_key' => '', 'secret_key' => '', 'webhook_secret' => ''],
                'supported_currencies' => ['USD', 'EUR', 'GBP', 'CAD', 'AUD', 'INR', 'NGN'],
                'transaction_fee_percentage' => 2.9,
                'transaction_fee_fixed' => 0.30,
                'display_order' => 2,
            ],
            [
                'name' => 'Paystack',
                'slug' => 'paystack',
                'display_name' => 'Paystack',
                'description' => 'Pay with card, bank transfer or USSD (Nigeria)',
                'is_active' => true,
                'is_test_mode' => true,
                'config' => ['public_key' => '', 'secret_key' => '', 'webhook_secret' => ''],
                'supported_currencies' => ['NGN', 'USD', 'GHS', 'ZAR', 'KES'],
                'transaction_fee_percentage' => 1.5,
                'transaction_fee_fixed' => 100.00,
                'display_order' => 3,
            ],
            [
                'name' => 'Flutterwave',
                'slug' => 'flutterwave',
                'display_name' => 'Flutterwave',
                'description' => 'Pay with multiple African payment methods',
                'is_active' => true,
                'is_test_mode' => true,
                'config' => ['public_key' => '', 'secret_key' => '', 'encryption_key' => '', 'webhook_secret' => ''],
                'supported_currencies' => ['NGN', 'USD', 'GHS', 'KES', 'UGX', 'ZAR', 'XAF', 'XOF'],
                'transaction_fee_percentage' => 1.4,
                'transaction_fee_fixed' => 0.00,
                'display_order' => 4,
            ],
            [
                'name' => 'Razorpay',
                'slug' => 'razorpay',
                'display_name' => 'Razorpay',
                'description' => 'Pay with multiple payment methods (India)',
                'is_active' => false,
                'is_test_mode' => true,
                'config' => ['key_id' => '', 'key_secret' => '', 'webhook_secret' => ''],
                'supported_currencies' => ['INR', 'USD', 'EUR', 'GBP'],
                'transaction_fee_percentage' => 2.0,
                'transaction_fee_fixed' => 0.00,
                'display_order' => 5,
            ],
        ];

        foreach ($gateways as $gateway) {
            \App\Models\PaymentGateway::create($gateway);
        }

        // Create email templates
        $emailTemplates = [
            [
                'name' => 'Welcome Email',
                'slug' => 'welcome',
                'subject' => 'Welcome to {{site_name}}!',
                'body' => "Hi {{name}},\n\nWelcome to {{site_name}}! Your account has been successfully created.\n\nEmail: {{email}}\n\nYou can login at: {{login_url}}\n\nBest regards,\n{{site_name}} Team",
                'variables' => 'name, email, site_name, login_url',
            ],
            [
                'name' => 'Invoice Created',
                'slug' => 'invoice_created',
                'subject' => 'New Invoice #{{invoice_id}} - {{site_name}}',
                'body' => "Hi {{name}},\n\nA new invoice #{{invoice_id}} has been generated for your account.\n\nAmount: {{amount}}\nDue Date: {{due_date}}\n\nYou can view and pay your invoice at: {{invoice_url}}\n\nBest regards,\n{{site_name}} Team",
                'variables' => 'name, invoice_id, amount, due_date, invoice_url, site_name',
            ],
            [
                'name' => 'Invoice Payment Received',
                'slug' => 'invoice_paid',
                'subject' => 'Payment Received for Invoice #{{invoice_id}}',
                'body' => "Hi {{name}},\n\nWe have received your payment for invoice #{{invoice_id}}.\n\nAmount Paid: {{amount}}\nPayment Method: {{payment_method}}\n\nThank you for your payment!\n\nBest regards,\n{{site_name}} Team",
                'variables' => 'name, invoice_id, amount, payment_method, site_name',
            ],
            [
                'name' => 'Payment Reminder',
                'slug' => 'payment_reminder',
                'subject' => 'Payment Reminder - Invoice #{{invoice_id}}',
                'body' => "Hi {{name}},\n\nThis is a friendly reminder that invoice #{{invoice_id}} is due on {{due_date}}.\n\nAmount Due: {{amount}}\n\nPlease make your payment at: {{invoice_url}}\n\nBest regards,\n{{site_name}} Team",
                'variables' => 'name, invoice_id, amount, due_date, invoice_url, site_name',
            ],
            [
                'name' => 'Service Activated',
                'slug' => 'service_activated',
                'subject' => 'Your Service Has Been Activated',
                'body' => "Hi {{name}},\n\nYour service '{{service_name}}' has been activated!\n\nService Details:\n- Domain: {{domain}}\n- Billing Cycle: {{billing_cycle}}\n- Next Due Date: {{next_due_date}}\n\nYou can manage your service at: {{service_url}}\n\nBest regards,\n{{site_name}} Team",
                'variables' => 'name, service_name, domain, billing_cycle, next_due_date, service_url, site_name',
            ],
            [
                'name' => 'Service Suspended',
                'slug' => 'service_suspended',
                'subject' => 'Service Suspended - Action Required',
                'body' => "Hi {{name}},\n\nYour service '{{service_name}}' has been suspended due to non-payment.\n\nTo reactivate your service, please pay the outstanding invoice: {{invoice_url}}\n\nBest regards,\n{{site_name}} Team",
                'variables' => 'name, service_name, invoice_url, site_name',
            ],
            [
                'name' => 'Ticket Reply',
                'slug' => 'ticket_reply',
                'subject' => 'New Reply to Your Ticket #{{ticket_id}}',
                'body' => "Hi {{name}},\n\nA new reply has been added to your support ticket #{{ticket_id}}.\n\nSubject: {{ticket_subject}}\n\nView the ticket at: {{ticket_url}}\n\nBest regards,\n{{site_name}} Support Team",
                'variables' => 'name, ticket_id, ticket_subject, ticket_url, site_name',
            ],
        ];

        foreach ($emailTemplates as $template) {
            EmailTemplate::create($template);
        }

        // Create default settings
        Setting::set('site_name', 'BillingCS', 'string');
        Setting::set('currency', 'USD', 'string');
        Setting::set('tax_rate', '0', 'int');
        Setting::set('enable_registration', '1', 'bool');
        Setting::set('active_template', 'default', 'string');

        // Create default template
        Template::create([
            'name' => 'default',
            'display_name' => 'Default Template',
            'description' => 'Default billing template',
            'is_active' => true,
        ]);
    }
}
