<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Domains table for client domain management
        Schema::create('domains', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('registrar_id')->nullable()->constrained('domain_registrars')->onDelete('set null');
            $table->string('domain_name')->unique();
            $table->date('registration_date');
            $table->date('expiry_date');
            $table->enum('status', ['active', 'pending', 'expired', 'cancelled', 'transferred'])->default('active');
            $table->decimal('renewal_price', 10, 2);
            $table->boolean('auto_renew')->default(false);
            $table->timestamps();
        });

        // Messages/notifications table for client messaging
        Schema::create('messages', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->string('subject');
            $table->text('message');
            $table->boolean('read')->default(false);
            $table->timestamp('read_at')->nullable();
            $table->timestamps();
        });

        // Account credits table for add funds
        Schema::create('account_credits', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('transaction_id')->nullable()->constrained()->onDelete('set null');
            $table->decimal('amount', 10, 2);
            $table->text('description')->nullable();
            $table->enum('type', ['deposit', 'withdrawal', 'credit', 'debit'])->default('deposit');
            $table->timestamps();
        });

        // Contacts table for additional client contacts
        Schema::create('contacts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->string('name');
            $table->string('email');
            $table->string('phone')->nullable();
            $table->string('position')->nullable();
            $table->boolean('is_primary')->default(false);
            $table->timestamps();
        });

        // Service cancellation requests
        Schema::create('cancellation_requests', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->foreignId('service_id')->constrained()->onDelete('cascade');
            $table->enum('type', ['immediate', 'end_of_billing'])->default('end_of_billing');
            $table->text('reason')->nullable();
            $table->enum('status', ['pending', 'approved', 'rejected'])->default('pending');
            $table->timestamp('processed_at')->nullable();
            $table->timestamps();
        });

        // Affiliates table
        Schema::create('affiliates', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->onDelete('cascade');
            $table->string('affiliate_code')->unique();
            $table->decimal('commission_rate', 5, 2)->default(10.00);
            $table->decimal('total_earnings', 10, 2)->default(0);
            $table->decimal('pending_earnings', 10, 2)->default(0);
            $table->decimal('paid_earnings', 10, 2)->default(0);
            $table->boolean('active')->default(true);
            $table->timestamps();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('affiliates');
        Schema::dropIfExists('cancellation_requests');
        Schema::dropIfExists('contacts');
        Schema::dropIfExists('account_credits');
        Schema::dropIfExists('messages');
        Schema::dropIfExists('domains');
    }
};
