<?php

namespace App\Modules\PaymentGateways\Paystack;

class PaystackGateway
{
    protected $config;
    protected $secretKey;

    public function __construct($config = [])
    {
        $this->config = $config;
        $this->secretKey = $config['secret_key'] ?? env('PAYSTACK_SECRET');
    }

    public function createPayment($invoice)
    {
        // Real Paystack API Transaction initialization
        // Documentation: https://paystack.com/docs/api/#transaction-initialize
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.paystack.co/transaction/initialize',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->secretKey,
                'Content-Type: application/json'
            ],
            CURLOPT_POSTFIELDS => json_encode([
                'amount' => round($invoice->total * 100), // Paystack uses kobo for NGN
                'currency' => strtoupper($invoice->currency ?? 'NGN'),
                'email' => $invoice->user->email,
                'reference' => 'INV-' . $invoice->id . '-' . time(),
                'callback_url' => url('/payment/paystack/callback'),
                'metadata' => [
                    'invoice_id' => $invoice->id,
                    'user_id' => $invoice->user_id,
                    'custom_fields' => [
                        [
                            'display_name' => 'Invoice Number',
                            'variable_name' => 'invoice_number',
                            'value' => $invoice->id
                        ]
                    ]
                ]
            ])
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 200 && $data['status']) {
            return [
                'success' => true,
                'authorization_url' => $data['data']['authorization_url'],
                'access_code' => $data['data']['access_code'],
                'reference' => $data['data']['reference'],
                'transaction_id' => $data['data']['reference'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Payment initialization failed',
        ];
    }

    public function verifyPayment($reference)
    {
        // Real Paystack API Transaction verification
        // Documentation: https://paystack.com/docs/api/#transaction-verify
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.paystack.co/transaction/verify/' . $reference,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->secretKey,
            ]
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 200 && $data['status']) {
            $transactionData = $data['data'];
            return [
                'success' => $transactionData['status'] === 'success',
                'status' => $transactionData['status'],
                'amount' => $transactionData['amount'] / 100,
                'currency' => $transactionData['currency'],
                'reference' => $transactionData['reference'],
                'paid_at' => $transactionData['paid_at'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Verification failed',
        ];
    }

    public function refundPayment($reference, $amount = null)
    {
        // Real Paystack API Refund creation
        // Documentation: https://paystack.com/docs/api/#refund-create
        
        $curl = curl_init();
        
        $postData = ['transaction' => $reference];
        if ($amount) {
            $postData['amount'] = round($amount * 100);
        }
        
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.paystack.co/refund',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->secretKey,
                'Content-Type: application/json'
            ],
            CURLOPT_POSTFIELDS => json_encode($postData)
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 200 && $data['status']) {
            return [
                'success' => true,
                'refund_id' => $data['data']['id'],
                'amount' => $data['data']['amount'] / 100,
                'status' => $data['data']['status'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Refund failed',
        ];
    }

    public function handleWebhook($payload, $signature)
    {
        // Real Paystack webhook signature verification
        // Documentation: https://paystack.com/docs/payments/webhooks/
        
        $webhookSecret = $this->config['webhook_secret'] ?? env('PAYSTACK_WEBHOOK_SECRET');
        
        $expectedSignature = hash_hmac('sha512', $payload, $webhookSecret);
        
        if ($signature === $expectedSignature) {
            $event = json_decode($payload, true);
            
            return [
                'success' => true,
                'event' => $event['event'],
                'data' => $event['data'],
            ];
        }

        return [
            'success' => false,
            'error' => 'Invalid signature',
        ];
    }
}
