<?php

namespace App\Modules\PaymentGateways\PayPal;

class PayPalGateway
{
    protected $config;
    protected $clientId;
    protected $secret;
    protected $baseUrl;

    public function __construct($config = [])
    {
        $this->config = $config;
        $this->clientId = $config['client_id'] ?? env('PAYPAL_CLIENT_ID');
        $this->secret = $config['secret'] ?? env('PAYPAL_SECRET');
        $mode = $config['mode'] ?? env('PAYPAL_MODE', 'sandbox');
        $this->baseUrl = $mode === 'live' 
            ? 'https://api-m.paypal.com' 
            : 'https://api-m.sandbox.paypal.com';
    }

    protected function getAccessToken()
    {
        // Real PayPal OAuth 2.0 token generation
        // Documentation: https://developer.paypal.com/api/rest/authentication/
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->baseUrl . '/v1/oauth2/token',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_USERPWD => $this->clientId . ':' . $this->secret,
            CURLOPT_POSTFIELDS => 'grant_type=client_credentials'
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        $data = json_decode($response, true);
        return $data['access_token'] ?? null;
    }

    public function createPayment($invoice)
    {
        // Real PayPal Orders API v2
        // Documentation: https://developer.paypal.com/docs/api/orders/v2/
        
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return ['success' => false, 'error' => 'Failed to get PayPal access token'];
        }

        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->baseUrl . '/v2/checkout/orders',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken,
            ],
            CURLOPT_POSTFIELDS => json_encode([
                'intent' => 'CAPTURE',
                'purchase_units' => [[
                    'reference_id' => 'INV-' . $invoice->id,
                    'amount' => [
                        'currency_code' => strtoupper($invoice->currency ?? 'USD'),
                        'value' => number_format($invoice->total, 2, '.', ''),
                    ],
                    'description' => 'Invoice #' . $invoice->id,
                ]],
                'application_context' => [
                    'return_url' => url('/payment/paypal/success'),
                    'cancel_url' => url('/payment/paypal/cancel'),
                ]
            ])
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 201 && isset($data['id'])) {
            $approveLink = collect($data['links'])->firstWhere('rel', 'approve');
            return [
                'success' => true,
                'order_id' => $data['id'],
                'approval_url' => $approveLink['href'] ?? '',
                'transaction_id' => $data['id'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Order creation failed',
        ];
    }

    public function verifyPayment($orderId)
    {
        // Real PayPal Order capture
        // Documentation: https://developer.paypal.com/docs/api/orders/v2/#orders_capture
        
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return ['success' => false, 'error' => 'Failed to get PayPal access token'];
        }

        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->baseUrl . '/v2/checkout/orders/' . $orderId . '/capture',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken,
            ],
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 201 && $data['status'] === 'COMPLETED') {
            $capture = $data['purchase_units'][0]['payments']['captures'][0];
            return [
                'success' => true,
                'status' => 'completed',
                'capture_id' => $capture['id'],
                'amount' => $capture['amount']['value'],
                'currency' => $capture['amount']['currency_code'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Payment capture failed',
        ];
    }

    public function refundPayment($captureId, $amount = null)
    {
        // Real PayPal Refund API
        // Documentation: https://developer.paypal.com/docs/api/payments/v2/#captures_refund
        
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return ['success' => false, 'error' => 'Failed to get PayPal access token'];
        }

        $curl = curl_init();
        
        $postData = [];
        if ($amount) {
            $postData['amount'] = [
                'value' => number_format($amount, 2, '.', ''),
                'currency_code' => 'USD',
            ];
        }
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->baseUrl . '/v2/payments/captures/' . $captureId . '/refund',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken,
            ],
            CURLOPT_POSTFIELDS => json_encode($postData)
        ]);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($httpCode === 201 && $data['status'] === 'COMPLETED') {
            return [
                'success' => true,
                'refund_id' => $data['id'],
                'status' => $data['status'],
                'amount' => $data['amount']['value'],
            ];
        }

        return [
            'success' => false,
            'error' => $data['message'] ?? 'Refund failed',
        ];
    }

    public function handleWebhook($payload, $headers)
    {
        // Real PayPal Webhook signature verification
        // Documentation: https://developer.paypal.com/docs/api-basics/notifications/webhooks/notification-messages/
        
        $webhookId = $this->config['webhook_id'] ?? env('PAYPAL_WEBHOOK_ID');
        $accessToken = $this->getAccessToken();
        
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->baseUrl . '/v1/notifications/verify-webhook-signature',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken,
            ],
            CURLOPT_POSTFIELDS => json_encode([
                'transmission_id' => $headers['PAYPAL-TRANSMISSION-ID'] ?? '',
                'transmission_time' => $headers['PAYPAL-TRANSMISSION-TIME'] ?? '',
                'cert_url' => $headers['PAYPAL-CERT-URL'] ?? '',
                'auth_algo' => $headers['PAYPAL-AUTH-ALGO'] ?? '',
                'transmission_sig' => $headers['PAYPAL-TRANSMISSION-SIG'] ?? '',
                'webhook_id' => $webhookId,
                'webhook_event' => json_decode($payload, true),
            ])
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        $data = json_decode($response, true);

        if ($data['verification_status'] === 'SUCCESS') {
            $event = json_decode($payload, true);
            return [
                'success' => true,
                'event_type' => $event['event_type'],
                'resource' => $event['resource'],
            ];
        }

        return [
            'success' => false,
            'error' => 'Webhook verification failed',
        ];
    }
}

