<?php

namespace App\Modules\DomainRegistrars\Namecheap;

class NamecheapRegistrar
{
    protected $config;
    protected $apiUser;
    protected $apiKey;
    protected $username;
    protected $baseUrl;

    public function __construct($config = [])
    {
        $this->config = $config;
        $this->apiUser = $config['api_user'] ?? env('NAMECHEAP_API_USER');
        $this->apiKey = $config['api_key'] ?? env('NAMECHEAP_API_KEY');
        $this->username = $config['username'] ?? env('NAMECHEAP_USERNAME');
        $sandbox = $config['sandbox'] ?? env('NAMECHEAP_SANDBOX', true);
        $this->baseUrl = $sandbox 
            ? 'https://api.sandbox.namecheap.com/xml.response'
            : 'https://api.namecheap.com/xml.response';
    }

    public function checkAvailability($domain)
    {
        // Real Namecheap API Domain Check
        // Documentation: https://www.namecheap.com/support/api/methods/domains/check/
        
        $params = [
            'ApiUser' => $this->apiUser,
            'ApiKey' => $this->apiKey,
            'UserName' => $this->username,
            'Command' => 'namecheap.domains.check',
            'ClientIp' => request()->ip(),
            'DomainList' => $domain,
        ];

        $url = $this->baseUrl . '?' . http_build_query($params);
        
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        $xml = simplexml_load_string($response);
        
        if ($xml && $xml['Status'] == 'OK') {
            $domainCheck = $xml->CommandResponse->DomainCheckResult;
            return [
                'success' => true,
                'available' => (string)$domainCheck['Available'] === 'true',
                'domain' => (string)$domainCheck['Domain'],
            ];
        }

        return [
            'success' => false,
            'error' => 'Domain check failed',
        ];
    }

    public function registerDomain($domain, $years, $contactInfo)
    {
        // Real Namecheap API Domain Registration
        // Documentation: https://www.namecheap.com/support/api/methods/domains/create/
        
        $params = [
            'ApiUser' => $this->apiUser,
            'ApiKey' => $this->apiKey,
            'UserName' => $this->username,
            'Command' => 'namecheap.domains.create',
            'ClientIp' => request()->ip(),
            'DomainName' => $domain,
            'Years' => $years,
            'RegistrantFirstName' => $contactInfo['first_name'],
            'RegistrantLastName' => $contactInfo['last_name'],
            'RegistrantAddress1' => $contactInfo['address'],
            'RegistrantCity' => $contactInfo['city'],
            'RegistrantStateProvince' => $contactInfo['state'],
            'RegistrantPostalCode' => $contactInfo['zip'],
            'RegistrantCountry' => $contactInfo['country'],
            'RegistrantPhone' => $contactInfo['phone'],
            'RegistrantEmailAddress' => $contactInfo['email'],
            'TechFirstName' => $contactInfo['first_name'],
            'TechLastName' => $contactInfo['last_name'],
            'TechAddress1' => $contactInfo['address'],
            'TechCity' => $contactInfo['city'],
            'TechStateProvince' => $contactInfo['state'],
            'TechPostalCode' => $contactInfo['zip'],
            'TechCountry' => $contactInfo['country'],
            'TechPhone' => $contactInfo['phone'],
            'TechEmailAddress' => $contactInfo['email'],
            'AdminFirstName' => $contactInfo['first_name'],
            'AdminLastName' => $contactInfo['last_name'],
            'AdminAddress1' => $contactInfo['address'],
            'AdminCity' => $contactInfo['city'],
            'AdminStateProvince' => $contactInfo['state'],
            'AdminPostalCode' => $contactInfo['zip'],
            'AdminCountry' => $contactInfo['country'],
            'AdminPhone' => $contactInfo['phone'],
            'AdminEmailAddress' => $contactInfo['email'],
            'AuxBillingFirstName' => $contactInfo['first_name'],
            'AuxBillingLastName' => $contactInfo['last_name'],
            'AuxBillingAddress1' => $contactInfo['address'],
            'AuxBillingCity' => $contactInfo['city'],
            'AuxBillingStateProvince' => $contactInfo['state'],
            'AuxBillingPostalCode' => $contactInfo['zip'],
            'AuxBillingCountry' => $contactInfo['country'],
            'AuxBillingPhone' => $contactInfo['phone'],
            'AuxBillingEmailAddress' => $contactInfo['email'],
        ];

        $url = $this->baseUrl . '?' . http_build_query($params);
        
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        $xml = simplexml_load_string($response);
        
        if ($xml && $xml['Status'] == 'OK') {
            $result = $xml->CommandResponse->DomainCreateResult;
            return [
                'success' => true,
                'domain' => (string)$result['Domain'],
                'registered' => (string)$result['Registered'] === 'true',
                'order_id' => (string)$result['OrderID'],
                'transaction_id' => (string)$result['TransactionID'],
            ];
        }

        return [
            'success' => false,
            'error' => $xml->Errors->Error ?? 'Domain registration failed',
        ];
    }

    public function renewDomain($domain, $years)
    {
        // Real Namecheap API Domain Renewal
        // Documentation: https://www.namecheap.com/support/api/methods/domains/renew/
        
        $params = [
            'ApiUser' => $this->apiUser,
            'ApiKey' => $this->apiKey,
            'UserName' => $this->username,
            'Command' => 'namecheap.domains.renew',
            'ClientIp' => request()->ip(),
            'DomainName' => $domain,
            'Years' => $years,
        ];

        $url = $this->baseUrl . '?' . http_build_query($params);
        
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        $xml = simplexml_load_string($response);
        
        if ($xml && $xml['Status'] == 'OK') {
            $result = $xml->CommandResponse->DomainRenewResult;
            return [
                'success' => true,
                'domain' => (string)$result['DomainName'],
                'renewed' => (string)$result['Renew'] === 'true',
                'order_id' => (string)$result['OrderID'],
                'transaction_id' => (string)$result['TransactionID'],
            ];
        }

        return [
            'success' => false,
            'error' => 'Domain renewal failed',
        ];
    }

    public function transferDomain($domain, $eppCode, $years)
    {
        // Real Namecheap API Domain Transfer
        // Documentation: https://www.namecheap.com/support/api/methods/domains/transfer/create/
        
        $params = [
            'ApiUser' => $this->apiUser,
            'ApiKey' => $this->apiKey,
            'UserName' => $this->username,
            'Command' => 'namecheap.domains.transfer.create',
            'ClientIp' => request()->ip(),
            'DomainName' => $domain,
            'Years' => $years,
            'EPPCode' => $eppCode,
        ];

        $url = $this->baseUrl . '?' . http_build_query($params);
        
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        $xml = simplexml_load_string($response);
        
        if ($xml && $xml['Status'] == 'OK') {
            $result = $xml->CommandResponse->DomainTransferCreateResult;
            return [
                'success' => true,
                'transfer_id' => (string)$result['Transfer'],
                'order_id' => (string)$result['OrderID'],
                'transaction_id' => (string)$result['TransactionID'],
            ];
        }

        return [
            'success' => false,
            'error' => 'Domain transfer failed',
        ];
    }
}
