<?php

namespace App\Http\Controllers\Admin\Support;

use App\Http\Controllers\Controller;
use App\Models\Ticket;
use App\Models\User;
use Illuminate\Http\Request;

class SupportController extends Controller
{
    public function overview()
    {
        $stats = [
            'total_tickets' => Ticket::count(),
            'open_tickets' => Ticket::where('status', 'open')->count(),
            'answered_tickets' => Ticket::where('status', 'answered')->count(),
            'customer_reply_tickets' => Ticket::where('status', 'customer-reply')->count(),
            'on_hold_tickets' => Ticket::where('status', 'on-hold')->count(),
            'in_progress_tickets' => Ticket::where('status', 'in-progress')->count(),
            'closed_tickets' => Ticket::where('status', 'closed')->count(),
            'avg_response_time' => '2.5 hours', // Calculate from ticket replies
            'satisfaction_rate' => '94%', // Calculate from feedback
        ];

        $recent_tickets = Ticket::with('user')
                                ->latest()
                                ->limit(10)
                                ->get();
        
        $ticket_stats_by_day = Ticket::selectRaw('DATE(created_at) as date, COUNT(*) as count')
                                    ->where('created_at', '>=', now()->subDays(7))
                                    ->groupBy('date')
                                    ->orderBy('date')
                                    ->get();
        
        return view('admin.support.overview', compact('stats', 'recent_tickets', 'ticket_stats_by_day'));
    }

    public function tickets(Request $request)
    {
        $query = Ticket::with('user');

        if ($request->has('status')) {
            $query->where('status', $request->get('status'));
        }

        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function($q) use ($search) {
                $q->where('subject', 'like', "%{$search}%")
                  ->orWhere('ticket_number', 'like', "%{$search}%")
                  ->orWhereHas('user', function($subQ) use ($search) {
                      $subQ->where('name', 'like', "%{$search}%")
                           ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        $tickets = $query->latest()->paginate(20);
        
        return view('admin.support.tickets.index', compact('tickets'));
    }

    public function flagged()
    {
        $tickets = Ticket::where('flagged', true)
                         ->with('user')
                         ->latest()
                         ->paginate(20);
        
        return view('admin.support.tickets.flagged', compact('tickets'));
    }

    public function open()
    {
        $tickets = Ticket::where('status', 'open')
                         ->with('user')
                         ->latest()
                         ->paginate(20);
        
        return view('admin.support.tickets.open', compact('tickets'));
    }

    public function answered()
    {
        $tickets = Ticket::where('status', 'answered')
                         ->with('user')
                         ->latest()
                         ->paginate(20);
        
        return view('admin.support.tickets.answered', compact('tickets'));
    }

    public function customerReply()
    {
        $tickets = Ticket::where('status', 'customer-reply')
                         ->with('user')
                         ->latest()
                         ->paginate(20);
        
        return view('admin.support.tickets.customer-reply', compact('tickets'));
    }

    public function onHold()
    {
        $tickets = Ticket::where('status', 'on-hold')
                         ->with('user')
                         ->latest()
                         ->paginate(20);
        
        return view('admin.support.tickets.on-hold', compact('tickets'));
    }

    public function inProgress()
    {
        $tickets = Ticket::where('status', 'in-progress')
                         ->with('user')
                         ->latest()
                         ->paginate(20);
        
        return view('admin.support.tickets.in-progress', compact('tickets'));
    }

    public function closed()
    {
        $tickets = Ticket::where('status', 'closed')
                         ->with('user')
                         ->latest()
                         ->paginate(20);
        
        return view('admin.support.tickets.closed', compact('tickets'));
    }

    public function predefinedReplies()
    {
        $replies = \App\Models\PredefinedReply::orderBy('name')->get();
        
        return view('admin.support.predefined-replies', compact('replies'));
    }

    public function announcements()
    {
        $announcements = \App\Models\Announcement::orderBy('published_at', 'desc')->paginate(20);
        
        return view('admin.support.announcements', compact('announcements'));
    }

    public function downloads()
    {
        $downloads = \App\Models\Download::orderBy('category')->orderBy('name')->paginate(20);
        
        return view('admin.support.downloads', compact('downloads'));
    }

    public function knowledgebase()
    {
        $categories = \App\Models\KnowledgebaseCategory::withCount('articles')->get();
        $articles = \App\Models\KnowledgebaseArticle::with('category')->latest()->paginate(20);
        
        return view('admin.support.knowledgebase', compact('categories', 'articles'));
    }

    public function networkIssues()
    {
        $issues = \App\Models\NetworkIssue::orderBy('created_at', 'desc')->paginate(20);
        
        $stats = [
            'open_issues' => \App\Models\NetworkIssue::where('status', 'open')->count(),
            'scheduled_issues' => \App\Models\NetworkIssue::where('status', 'scheduled')->count(),
            'resolved_issues' => \App\Models\NetworkIssue::where('status', 'resolved')->count(),
        ];
        
        return view('admin.support.network-issues.index', compact('issues', 'stats'));
    }
}
