<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Mail;

class SettingController extends Controller
{
    public function index()
    {
        return view('admin.settings.index');
    }

    // General Settings
    public function general()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.general', compact('settings'));
    }

    public function updateGeneral(Request $request)
    {
        $validated = $request->validate([
            'site_name' => 'required|string|max:255',
            'site_url' => 'required|url',
            'site_email' => 'nullable|email|max:255',
            'site_address' => 'nullable|string|max:500',
            'site_logo' => 'nullable|url',
            'currency' => 'required|string|max:10',
            'tax_rate' => 'nullable|numeric|min:0|max:100',
            'enable_registration' => 'nullable|boolean',
            'maintenance_mode' => 'nullable|boolean',
            'auto_invoice_generation' => 'nullable|boolean',
            'maintenance_message' => 'nullable|string|max:1000',
            'maintenance_redirect_url' => 'nullable|url',
        ]);

        foreach ($validated as $key => $value) {
            // Convert checkbox values
            if (in_array($key, ['enable_registration', 'maintenance_mode', 'auto_invoice_generation'])) {
                Setting::set($key, $value ? '1' : '0', 'bool');
            } else {
                Setting::set($key, $value ?? '');
            }
        }

        // Handle unchecked checkboxes
        $checkboxes = ['enable_registration', 'maintenance_mode', 'auto_invoice_generation'];
        foreach ($checkboxes as $checkbox) {
            if (! $request->has($checkbox)) {
                Setting::set($checkbox, '0', 'bool');
            }
        }

        // Clear cached settings so changes are reflected immediately
        Cache::forget('site_settings');
        Cache::forget('maintenance_mode');
        Cache::forget('maintenance_redirect_url');
        Cache::forget('maintenance_settings');

        return back()->with('success', 'General settings updated successfully');
    }

    // Email Settings
    public function email()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.email', compact('settings'));
    }

    public function updateEmail(Request $request)
    {
        $validated = $request->validate([
            'mail_mailer' => 'required|string|in:smtp,sendmail,log',
            'mail_host' => 'required|string|max:255',
            'mail_port' => 'required|integer|min:1|max:65535',
            'mail_encryption' => 'nullable|string|in:tls,ssl',
            'mail_username' => 'nullable|string|max:255',
            'mail_password' => 'nullable|string',
            'mail_from_name' => 'required|string|max:255',
            'mail_from_address' => 'required|email|max:255',
            'test_email_recipient' => 'nullable|email|max:255',
        ]);

        foreach ($validated as $key => $value) {
            // Don't update password if it's empty (keep existing)
            if ($key === 'mail_password' && empty($value)) {
                continue;
            }
            Setting::set($key, $value ?? '');
        }

        // Update Laravel's mail config dynamically
        $this->updateMailConfig();

        return back()->with('success', 'Email settings updated successfully');
    }

    public function testEmail(Request $request)
    {
        $validated = $request->validate([
            'mail_mailer' => 'required|string',
            'mail_host' => 'required|string',
            'mail_port' => 'required|integer',
            'mail_encryption' => 'nullable|string',
            'mail_username' => 'nullable|string',
            'mail_password' => 'nullable|string',
            'mail_from_name' => 'required|string',
            'mail_from_address' => 'required|email',
            'test_email_recipient' => 'nullable|email',
        ]);

        try {
            // Use SMTP username as the From address to avoid sender verification errors
            // Most SMTP servers require the From address to match the authenticated username
            $fromAddress = ! empty($validated['mail_username'])
                ? $validated['mail_username']
                : $validated['mail_from_address'];

            // Determine test email recipient
            // Priority: 1. Custom test recipient, 2. Site email (SMTP username), 3. From address
            $testRecipient = $validated['test_email_recipient']
                ?? $fromAddress;

            // Temporarily update mail config with form values
            config([
                'mail.default' => $validated['mail_mailer'],
                'mail.mailers.smtp.host' => $validated['mail_host'],
                'mail.mailers.smtp.port' => $validated['mail_port'],
                'mail.mailers.smtp.encryption' => $validated['mail_encryption'] ?? null,
                'mail.mailers.smtp.username' => $validated['mail_username'] ?? null,
                'mail.mailers.smtp.password' => $validated['mail_password'] ?? Setting::get('mail_password'),
                'mail.from.address' => $fromAddress,
                'mail.from.name' => $validated['mail_from_name'],
            ]);

            // Send test email using imported Mail facade
            // Explicitly set the from address and replyTo to match SMTP username for compliance
            Mail::raw(
                'This is a test email from '.$validated['mail_from_name'].'. If you received this, your email settings are configured correctly!',
                function ($message) use ($testRecipient, $validated, $fromAddress) {
                    $message->to($testRecipient)
                        ->subject('Test Email - '.$validated['mail_from_name'])
                        ->from($fromAddress, $validated['mail_from_name'])
                        ->replyTo($fromAddress, $validated['mail_from_name']);
                }
            );

            // If we reach here without exception, the email was accepted by the SMTP server
            return response()->json([
                'success' => true,
                'message' => 'Test email sent successfully to '.$testRecipient,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send test email: '.$e->getMessage(),
            ], 400);
        }
    }

    /**
     * Update Laravel's mail configuration from database settings
     */
    private function updateMailConfig()
    {
        $mailer = Setting::get('mail_mailer', 'smtp');
        $mailUsername = Setting::get('mail_username');
        $mailFromAddress = Setting::get('mail_from_address', 'noreply@example.com');

        // Use SMTP username as from address if configured to avoid sender verification errors
        // Most SMTP servers require the From address to match the authenticated username
        $fromAddress = ! empty($mailUsername) ? $mailUsername : $mailFromAddress;

        config([
            'mail.default' => $mailer,
            'mail.mailers.smtp.host' => Setting::get('mail_host', '127.0.0.1'),
            'mail.mailers.smtp.port' => Setting::get('mail_port', 587),
            'mail.mailers.smtp.encryption' => Setting::get('mail_encryption', 'tls'),
            'mail.mailers.smtp.username' => $mailUsername,
            'mail.mailers.smtp.password' => Setting::get('mail_password'),
            'mail.from.address' => $fromAddress,
            'mail.from.name' => Setting::get('mail_from_name', config('app.name')),
        ]);
    }

    // Payment Gateway Settings
    public function payment()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.payment', compact('settings'));
    }

    public function updatePayment(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::set($key, $value);
        }

        return back()->with('success', 'Payment settings updated successfully');
    }

    // User & Roles Settings
    public function users()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.users', compact('settings'));
    }

    public function updateUsers(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::set($key, $value);
        }

        return back()->with('success', 'User settings updated successfully');
    }

    // Product Settings
    public function products()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.products', compact('settings'));
    }

    public function updateProducts(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::set($key, $value);
        }

        return back()->with('success', 'Product settings updated successfully');
    }

    // Automation Settings
    public function automation()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.automation', compact('settings'));
    }

    public function updateAutomation(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::set($key, $value);
        }

        return back()->with('success', 'Automation settings updated successfully');
    }

    // Support Settings
    public function support()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.support', compact('settings'));
    }

    public function updateSupport(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::set($key, $value);
        }

        return back()->with('success', 'Support settings updated successfully');
    }

    // API Settings
    public function api()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.api', compact('settings'));
    }

    public function updateApi(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::set($key, $value);
        }

        return back()->with('success', 'API settings updated successfully');
    }

    // Security Settings
    public function security()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.security', compact('settings'));
    }

    public function updateSecurity(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::set($key, $value);
        }

        return back()->with('success', 'Security settings updated successfully');
    }

    // Module Settings
    public function modules()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.modules', compact('settings'));
    }

    public function updateModules(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::set($key, $value);
        }

        return back()->with('success', 'Module settings updated successfully');
    }

    // Domain Settings
    public function domains()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.domains', compact('settings'));
    }

    public function updateDomains(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::set($key, $value);
        }

        return back()->with('success', 'Domain settings updated successfully');
    }

    // Localization Settings
    public function localization()
    {
        $settings = Setting::all()->pluck('value', 'key');

        return view('admin.settings.categories.localization', compact('settings'));
    }

    public function updateLocalization(Request $request)
    {
        foreach ($request->except('_token') as $key => $value) {
            Setting::set($key, $value);
        }

        return back()->with('success', 'Localization settings updated successfully');
    }

    // Server Settings
    public function servers()
    {
        $servers = \App\Models\Server::paginate(15);

        return view('admin.settings.categories.servers', compact('servers'));
    }

    public function storeServer(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'hostname' => 'required|string|max:255',
            'ip_address' => 'required|ip',
            'type' => 'required|in:cpanel,directadmin,plesk,custom',
            'username' => 'required|string|max:255',
            'password' => 'required|string',
            'port' => 'required|integer',
            'secure' => 'boolean',
            'active' => 'boolean',
            'max_accounts' => 'nullable|integer|min:0',
            'nameserver1' => 'nullable|string',
            'nameserver2' => 'nullable|string',
            'nameserver3' => 'nullable|string',
            'nameserver4' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        // Encrypt the password
        $validated['password'] = $this->encryptPassword($validated['password']);
        $validated['secure'] = $request->has('secure');
        $validated['active'] = $request->has('active');

        \App\Models\Server::create($validated);

        return redirect()->route('admin.settings.servers')->with('success', 'Server added successfully');
    }

    public function editServer(\App\Models\Server $server)
    {
        return view('admin.settings.categories.servers-edit', compact('server'));
    }

    public function updateServer(Request $request, \App\Models\Server $server)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'hostname' => 'required|string|max:255',
            'ip_address' => 'required|ip',
            'type' => 'required|in:cpanel,directadmin,plesk,custom',
            'username' => 'required|string|max:255',
            'password' => 'nullable|string',
            'port' => 'required|integer',
            'secure' => 'boolean',
            'active' => 'boolean',
            'max_accounts' => 'nullable|integer|min:0',
            'nameserver1' => 'nullable|string',
            'nameserver2' => 'nullable|string',
            'nameserver3' => 'nullable|string',
            'nameserver4' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        // Only update password if provided
        if ($request->filled('password')) {
            $validated['password'] = $this->encryptPassword($validated['password']);
        } else {
            unset($validated['password']);
        }

        $validated['secure'] = $request->has('secure');
        $validated['active'] = $request->has('active');

        $server->update($validated);

        return redirect()->route('admin.settings.servers')->with('success', 'Server updated successfully');
    }

    public function destroyServer(\App\Models\Server $server)
    {
        $server->delete();

        return redirect()->route('admin.settings.servers')->with('success', 'Server deleted successfully');
    }

    public function testServerConnection(Request $request, ?\App\Models\Server $server = null)
    {
        // If server ID is provided, use existing server connection details
        if ($server) {
            try {
                $url = ($server->secure ? 'https://' : 'http://').$server->hostname.':'.$server->port;
                $result = $this->performConnectionTest($url);

                return response()->json([
                    'success' => $result['success'],
                    'message' => $result['success'] ? 'Connection test successful! Server is reachable.' : 'Could not connect to server.',
                    'url' => $url,
                ]);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Connection test failed: '.$e->getMessage(),
                ]);
            }
        }

        // Test connection with provided details (for new servers)
        $validated = $request->validate([
            'hostname' => 'required|string',
            'port' => 'required|integer',
            'username' => 'required|string',
            'password' => 'required|string',
            'type' => 'required|string',
            'secure' => 'boolean',
        ]);

        try {
            $url = ($validated['secure'] ? 'https://' : 'http://').$validated['hostname'].':'.$validated['port'];
            $result = $this->performConnectionTest($url);

            return response()->json([
                'success' => $result['success'],
                'message' => $result['success'] ? 'Connection test successful! Server is reachable on '.$url : 'Could not connect to server at '.$url.'. Please verify hostname, port, and firewall settings.',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Connection test failed: '.$e->getMessage(),
            ]);
        }
    }

    /**
     * Encrypt a password for secure storage
     */
    private function encryptPassword(string $password): string
    {
        return \Illuminate\Support\Facades\Crypt::encryptString($password);
    }

    /**
     * Perform a connection test to a server URL
     */
    private function performConnectionTest(string $url): array
    {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);

        // Check if SSL verification should be disabled (configurable via environment)
        $verifySsl = config('app.verify_server_ssl', true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, $verifySsl);

        if (! $verifySsl) {
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        }

        curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return [
            'success' => $httpCode > 0,
            'http_code' => $httpCode,
        ];
    }
}
