<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Server;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;

class ServerController extends Controller
{
    public function index()
    {
        $servers = Server::paginate(config('app.items_per_page'));
        return view('admin.servers.index', compact('servers'));
    }

    public function create()
    {
        return view('admin.servers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'hostname' => 'required|string|max:255',
            'ip_address' => 'required|ip',
            'type' => 'required|in:cpanel,directadmin,plesk,custom',
            'username' => 'required|string|max:255',
            'password' => 'required|string',
            'port' => 'required|integer',
            'secure' => 'boolean',
            'active' => 'boolean',
            'max_accounts' => 'nullable|integer|min:0',
            'nameserver1' => 'nullable|string',
            'nameserver2' => 'nullable|string',
            'nameserver3' => 'nullable|string',
            'nameserver4' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        // Encrypt the password
        $validated['password'] = Crypt::encryptString($validated['password']);
        $validated['secure'] = $request->has('secure');
        $validated['active'] = $request->has('active');

        Server::create($validated);

        return redirect()->route('admin.servers.index')->with('success', 'Server added successfully');
    }

    public function show(Server $server)
    {
        return view('admin.servers.show', compact('server'));
    }

    public function edit(Server $server)
    {
        return view('admin.servers.edit', compact('server'));
    }

    public function update(Request $request, Server $server)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'hostname' => 'required|string|max:255',
            'ip_address' => 'required|ip',
            'type' => 'required|in:cpanel,directadmin,plesk,custom',
            'username' => 'required|string|max:255',
            'password' => 'nullable|string',
            'port' => 'required|integer',
            'secure' => 'boolean',
            'active' => 'boolean',
            'max_accounts' => 'nullable|integer|min:0',
            'nameserver1' => 'nullable|string',
            'nameserver2' => 'nullable|string',
            'nameserver3' => 'nullable|string',
            'nameserver4' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        // Only update password if provided
        if ($request->filled('password')) {
            $validated['password'] = Crypt::encryptString($validated['password']);
        } else {
            unset($validated['password']);
        }

        $validated['secure'] = $request->has('secure');
        $validated['active'] = $request->has('active');

        $server->update($validated);

        return redirect()->route('admin.servers.index')->with('success', 'Server updated successfully');
    }

    public function destroy(Server $server)
    {
        $server->delete();
        return redirect()->route('admin.servers.index')->with('success', 'Server deleted successfully');
    }

    public function testConnection(Server $server)
    {
        // Test connection to server
        // This would integrate with actual cPanel/DirectAdmin API
        return response()->json([
            'success' => true,
            'message' => 'Connection test successful',
            'url' => $server->getConnectionUrl(),
        ]);
    }
}
