<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Service;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Ticket;
use App\Models\Product;
use App\Models\Quote;
use Illuminate\Http\Request;

class QuickActionsController extends Controller
{
    public function newClient()
    {
        return view('admin.new.client');
    }

    public function storeClient(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8',
            'company' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'zip' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
        ]);

        $validated['password'] = bcrypt($validated['password']);
        $validated['role'] = 'client';

        $user = User::create($validated);

        return redirect()->route('admin.clients.show', $user->id)
                        ->with('success', 'Client created successfully');
    }

    public function newOrder()
    {
        $clients = User::where('role', 'client')->get();
        $products = Product::where('active', true)->with('pricing')->get();
        
        return view('admin.new.order', compact('clients', 'products'));
    }

    public function storeOrder(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'product_id' => 'required|exists:products,id',
            'billing_cycle' => 'required|in:monthly,quarterly,semi-annually,annually',
        ]);

        $product = Product::findOrFail($validated['product_id']);
        $pricing = $product->getPriceForCycle($validated['billing_cycle']);

        $service = Service::create([
            'user_id' => $validated['user_id'],
            'product_id' => $validated['product_id'],
            'name' => $product->name,
            'type' => $product->type,
            'description' => $product->description,
            'price' => $pricing ? $pricing->price : 0,
            'billing_cycle' => $validated['billing_cycle'],
            'status' => 'pending',
            'next_due_date' => now()->addMonth(),
        ]);

        return redirect()->route('admin.services.show', $service->id)
                        ->with('success', 'Order created successfully');
    }

    public function newInvoice()
    {
        $clients = User::where('role', 'client')->get();
        
        return view('admin.new.invoice', compact('clients'));
    }

    public function storeInvoice(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'date' => 'required|date',
            'due_date' => 'required|date|after:date',
            'items' => 'required|array|min:1',
            'items.*.description' => 'required|string',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
        ]);

        // Generate invoice number
        $lastInvoice = Invoice::latest('id')->first();
        $nextNumber = $lastInvoice ? (int)substr($lastInvoice->invoice_number, 4) + 1 : 1;
        $invoiceNumber = 'INV-' . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);

        // Calculate totals
        $subtotal = 0;
        foreach ($validated['items'] as $item) {
            $subtotal += $item['quantity'] * $item['unit_price'];
        }
        $tax = $subtotal * 0.10; // 10% tax
        $total = $subtotal + $tax;

        // Create invoice
        $invoice = Invoice::create([
            'user_id' => $validated['user_id'],
            'invoice_number' => $invoiceNumber,
            'date' => $validated['date'],
            'due_date' => $validated['due_date'],
            'subtotal' => $subtotal,
            'tax' => $tax,
            'total' => $total,
            'status' => 'unpaid',
        ]);

        // Create invoice items
        foreach ($validated['items'] as $item) {
            $itemTotal = $item['quantity'] * $item['unit_price'];
            InvoiceItem::create([
                'invoice_id' => $invoice->id,
                'description' => $item['description'],
                'quantity' => $item['quantity'],
                'unit_price' => $item['unit_price'],
                'total' => $itemTotal,
            ]);
        }

        return redirect()->route('admin.invoices.show', $invoice->id)
                        ->with('success', 'Invoice created successfully');
    }

    public function newQuote()
    {
        $clients = User::where('role', 'client')->get();
        $products = Product::where('active', true)->get();
        
        return view('admin.new.quote', compact('clients', 'products'));
    }

    public function storeQuote(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'subject' => 'required|string|max:255',
            'date' => 'required|date',
            'expiry_date' => 'required|date|after:date',
            'subtotal' => 'required|numeric|min:0',
            'tax' => 'nullable|numeric|min:0',
            'total' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        // Generate quote number
        $lastQuote = Quote::latest('id')->first();
        $nextNumber = $lastQuote ? (int)substr($lastQuote->quote_number, 3) + 1 : 1;
        $validated['quote_number'] = 'QT-' . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);
        $validated['status'] = 'pending';

        $quote = Quote::create($validated);

        return redirect()->route('admin.billing.quotes')
                        ->with('success', 'Quote created successfully');
    }

    public function newTicket()
    {
        $clients = User::where('role', 'client')->get();
        
        return view('admin.new.ticket', compact('clients'));
    }

    public function storeTicket(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'subject' => 'required|string|max:255',
            'priority' => 'required|in:low,medium,high',
            'message' => 'required|string',
        ]);

        // Generate ticket number
        $lastTicket = Ticket::latest('id')->first();
        $nextNumber = $lastTicket ? (int)substr($lastTicket->ticket_number, 4) + 1 : 1;
        $validated['ticket_number'] = 'TKT-' . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);
        $validated['status'] = 'open';

        $ticket = Ticket::create($validated);

        // Create initial ticket reply (message)
        \App\Models\TicketReply::create([
            'ticket_id' => $ticket->id,
            'user_id' => $validated['user_id'],
            'message' => $validated['message'],
        ]);

        return redirect()->route('admin.tickets.show', $ticket->id)
                        ->with('success', 'Ticket created successfully');
    }
}
