<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductPricing;
use App\Models\Server;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function index()
    {
        $products = Product::with('server')->paginate(config('app.items_per_page'));
        return view('admin.products.index', compact('products'));
    }

    public function create()
    {
        $servers = Server::where('active', true)->get();
        return view('admin.products.create', compact('servers'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:hosting,domain,vps,dedicated',
            'server_id' => 'nullable|exists:servers,id',
            'active' => 'boolean',
            'features' => 'nullable|array',
            'pricing' => 'required|array',
            'pricing.*.billing_cycle' => 'required|string',
            'pricing.*.price' => 'required|numeric|min:0',
            'pricing.*.setup_fee' => 'nullable|numeric|min:0',
        ]);

        $validated['active'] = $request->has('active');
        $pricing = $validated['pricing'];
        unset($validated['pricing']);

        $product = Product::create($validated);

        // Create pricing
        foreach ($pricing as $priceData) {
            ProductPricing::create([
                'product_id' => $product->id,
                'billing_cycle' => $priceData['billing_cycle'],
                'price' => $priceData['price'],
                'setup_fee' => $priceData['setup_fee'] ?? 0,
            ]);
        }

        return redirect()->route('admin.products.index')->with('success', 'Product created successfully');
    }

    public function show(Product $product)
    {
        $product->load('pricing', 'server');
        return view('admin.products.show', compact('product'));
    }

    public function edit(Product $product)
    {
        $servers = Server::where('active', true)->get();
        $product->load('pricing');
        return view('admin.products.edit', compact('product', 'servers'));
    }

    public function update(Request $request, Product $product)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:hosting,domain,vps,dedicated',
            'server_id' => 'nullable|exists:servers,id',
            'active' => 'boolean',
            'features' => 'nullable|array',
            'pricing' => 'required|array',
            'pricing.*.billing_cycle' => 'required|string',
            'pricing.*.price' => 'required|numeric|min:0',
            'pricing.*.setup_fee' => 'nullable|numeric|min:0',
        ]);

        $validated['active'] = $request->has('active');
        $pricing = $validated['pricing'];
        unset($validated['pricing']);

        $product->update($validated);

        // Delete old pricing and create new ones
        $product->pricing()->delete();
        foreach ($pricing as $priceData) {
            ProductPricing::create([
                'product_id' => $product->id,
                'billing_cycle' => $priceData['billing_cycle'],
                'price' => $priceData['price'],
                'setup_fee' => $priceData['setup_fee'] ?? 0,
            ]);
        }

        return redirect()->route('admin.products.index')->with('success', 'Product updated successfully');
    }

    public function destroy(Product $product)
    {
        $product->delete();
        return redirect()->route('admin.products.index')->with('success', 'Product deleted successfully');
    }
}
