<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use ZipArchive;

class ModuleManagerController extends Controller
{
    public function index()
    {
        $modules = $this->getInstalledModules();
        return view('admin.modules.index', compact('modules'));
    }

    public function upload(Request $request)
    {
        $request->validate([
            'module_file' => 'required|file|mimes:zip|max:10240',
        ]);

        $file = $request->file('module_file');
        $zip = new ZipArchive;
        
        if ($zip->open($file->getPathname()) === TRUE) {
            // Extract to temporary directory
            $tempPath = storage_path('app/temp/' . uniqid());
            $zip->extractTo($tempPath);
            $zip->close();

            // Validate module.json
            $moduleJsonPath = $tempPath . '/module.json';
            if (!file_exists($moduleJsonPath)) {
                File::deleteDirectory($tempPath);
                return back()->with('error', 'Invalid module: module.json not found');
            }

            $moduleConfig = json_decode(file_get_contents($moduleJsonPath), true);
            if (!$moduleConfig || !isset($moduleConfig['name'], $moduleConfig['type'])) {
                File::deleteDirectory($tempPath);
                return back()->with('error', 'Invalid module.json format');
            }

            // Determine module destination based on type
            $destinationPath = app_path('Modules/' . ucfirst($moduleConfig['type']) . '/' . $moduleConfig['name']);
            
            // Check if module already exists
            if (File::exists($destinationPath)) {
                File::deleteDirectory($tempPath);
                return back()->with('error', 'Module already exists');
            }

            // Move module to final location
            File::moveDirectory($tempPath, $destinationPath);

            // Register module in database
            \App\Models\ModuleModel::create([
                'name' => $moduleConfig['name'],
                'slug' => \Illuminate\Support\Str::slug($moduleConfig['name']),
                'version' => $moduleConfig['version'] ?? '1.0.0',
                'type' => $moduleConfig['type'],
                'status' => 'inactive',
                'config' => json_encode($moduleConfig),
            ]);

            return redirect()->route('admin.modules.index')
                ->with('success', 'Module uploaded and installed successfully');
        }

        return back()->with('error', 'Failed to extract module ZIP file');
    }

    public function toggle($id)
    {
        $module = \App\Models\ModuleModel::findOrFail($id);
        $module->status = $module->status === 'active' ? 'inactive' : 'active';
        $module->save();

        return back()->with('success', 'Module status updated');
    }

    public function uninstall($id)
    {
        $module = \App\Models\ModuleModel::findOrFail($id);
        
        // Delete module files
        $config = json_decode($module->config, true);
        $modulePath = app_path('Modules/' . ucfirst($config['type']) . '/' . $module->name);
        
        if (File::exists($modulePath)) {
            File::deleteDirectory($modulePath);
        }

        // Delete from database
        $module->delete();

        return redirect()->route('admin.modules.index')
            ->with('success', 'Module uninstalled successfully');
    }

    private function getInstalledModules()
    {
        return \App\Models\ModuleModel::orderBy('name')->get();
    }
}
