<?php

namespace App\Http\Controllers\Admin\Clients;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Service;
use App\Models\Invoice;
use App\Models\Ticket;
use Illuminate\Http\Request;

class ClientController extends Controller
{
    public function index(Request $request)
    {
        $query = User::where('role', 'client');

        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('company', 'like', "%{$search}%");
            });
        }

        $clients = $query->withCount(['services', 'invoices', 'tickets'])
                         ->with(['services' => function($q) {
                             $q->where('status', 'active');
                         }])
                         ->latest()
                         ->paginate(20);

        // Calculate total revenue per client
        foreach ($clients as $client) {
            $client->total_revenue = $client->invoices()->where('status', 'paid')->sum('total');
        }

        return view('admin.clients.index', compact('clients'));
    }

    public function show($id)
    {
        $client = User::where('role', 'client')
                      ->withCount(['services', 'invoices', 'tickets'])
                      ->findOrFail($id);
        
        $services = $client->services()->with('product')->latest()->paginate(10, ['*'], 'services_page');
        $invoices = $client->invoices()->latest()->paginate(10, ['*'], 'invoices_page');
        $tickets = $client->tickets()->latest()->paginate(10, ['*'], 'tickets_page');
        
        // Calculate statistics
        $stats = [
            'total_services' => $client->services()->count(),
            'active_services' => $client->services()->where('status', 'active')->count(),
            'total_revenue' => $client->invoices()->where('status', 'paid')->sum('total'),
            'unpaid_invoices' => $client->invoices()->where('status', 'unpaid')->sum('total'),
            'open_tickets' => $client->tickets()->where('status', 'open')->count(),
        ];
        
        return view('admin.clients.show', compact('client', 'services', 'invoices', 'tickets', 'stats'));
    }

    public function products()
    {
        $services = Service::with(['user', 'product'])
                           ->latest()
                           ->paginate(20);
        
        $stats = [
            'total_services' => Service::count(),
            'active_services' => Service::where('status', 'active')->count(),
            'pending_services' => Service::where('status', 'pending')->count(),
            'suspended_services' => Service::where('status', 'suspended')->count(),
        ];
        
        return view('admin.clients.products', compact('services', 'stats'));
    }

    public function addons()
    {
        // Get services with addon products
        $addons = Service::whereHas('product', function($q) {
            $q->where('type', 'addon');
        })->with(['user', 'product'])->latest()->paginate(20);
        
        return view('admin.clients.addons', compact('addons'));
    }

    public function domains()
    {
        // Get domain services
        $domains = Service::where('type', 'domain')
                          ->with(['user', 'product'])
                          ->latest()
                          ->paginate(20);
        
        $stats = [
            'total_domains' => Service::where('type', 'domain')->count(),
            'active_domains' => Service::where('type', 'domain')->where('status', 'active')->count(),
            'expiring_soon' => Service::where('type', 'domain')
                                     ->where('status', 'active')
                                     ->where('next_due_date', '<=', now()->addDays(30))
                                     ->count(),
        ];
        
        return view('admin.clients.domains', compact('domains', 'stats'));
    }

    public function cancellations()
    {
        $cancellations = Service::whereIn('status', ['cancelled', 'pending_cancellation'])
                                ->with(['user', 'product'])
                                ->latest()
                                ->paginate(20);
        
        $stats = [
            'pending_cancellations' => Service::where('status', 'pending_cancellation')->count(),
            'cancelled_services' => Service::where('status', 'cancelled')->count(),
            'cancellations_this_month' => Service::where('status', 'cancelled')
                                                 ->whereMonth('updated_at', now()->month)
                                                 ->count(),
        ];
        
        return view('admin.clients.cancellations', compact('cancellations', 'stats'));
    }

    public function affiliates()
    {
        $affiliates = \App\Models\Affiliate::with('user')->paginate(20);
        
        $stats = [
            'total_affiliates' => \App\Models\Affiliate::count(),
            'active_affiliates' => \App\Models\Affiliate::where('status', 'active')->count(),
            'total_referrals' => \App\Models\Affiliate::sum('referrals_count'),
            'total_commissions' => \App\Models\Affiliate::sum('total_earned'),
        ];
        
        return view('admin.clients.affiliates', compact('affiliates', 'stats'));
    }
}
