<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invoice;
use App\Models\Service;
use App\Models\User;
use Carbon\Carbon;

class ProcessDailyBilling extends Command
{
    protected $signature = 'billing:process-daily';
    protected $description = 'Process daily billing tasks: generate recurring invoices, send reminders, suspend services';

    public function handle()
    {
        $this->info('Starting daily billing process...');

        // 1. Generate recurring invoices
        $this->generateRecurringInvoices();

        // 2. Send payment reminders
        $this->sendPaymentReminders();

        // 3. Suspend overdue services
        $this->suspendOverdueServices();

        // 4. Terminate cancelled services
        $this->terminateCancelledServices();

        $this->info('Daily billing process completed successfully!');
    }

    private function generateRecurringInvoices()
    {
        $this->info('Generating recurring invoices...');

        $services = Service::where('status', 'active')
            ->whereDate('next_due_date', '<=', now()->addDays(7))
            ->get();

        foreach ($services as $service) {
            // Check if invoice already exists for this period
            $existingInvoice = Invoice::where('user_id', $service->user_id)
                ->where('status', 'unpaid')
                ->whereHas('items', function($query) use ($service) {
                    $query->where('description', 'like', '%' . $service->name . '%');
                })
                ->exists();

            if (!$existingInvoice) {
                // Create new invoice
                $invoice = Invoice::create([
                    'user_id' => $service->user_id,
                    'total' => $service->price,
                    'status' => 'unpaid',
                    'due_date' => $service->next_due_date,
                    'currency' => setting('currency', 'USD'),
                ]);

                // Create invoice item
                $invoice->items()->create([
                    'description' => $service->name . ' - ' . ucfirst($service->billing_cycle) . ' Billing',
                    'quantity' => 1,
                    'unit_price' => $service->price,
                    'total' => $service->price,
                ]);

                $this->info("Generated invoice #{$invoice->id} for service #{$service->id}");

                // Update next due date
                $service->next_due_date = $this->calculateNextDueDate($service->billing_cycle, $service->next_due_date);
                $service->save();
            }
        }
    }

    private function sendPaymentReminders()
    {
        $this->info('Sending payment reminders...');

        // Reminder 7 days before due
        $invoices7days = Invoice::where('status', 'unpaid')
            ->whereDate('due_date', '=', now()->addDays(7))
            ->get();

        foreach ($invoices7days as $invoice) {
            // Send email reminder (implement mail sending)
            $this->info("Sent 7-day reminder for invoice #{$invoice->id}");
        }

        // Reminder 3 days before due
        $invoices3days = Invoice::where('status', 'unpaid')
            ->whereDate('due_date', '=', now()->addDays(3))
            ->get();

        foreach ($invoices3days as $invoice) {
            // Send email reminder
            $this->info("Sent 3-day reminder for invoice #{$invoice->id}");
        }

        // Overdue reminder
        $overdueInvoices = Invoice::where('status', 'unpaid')
            ->whereDate('due_date', '<', now())
            ->whereDate('due_date', '>=', now()->subDays(7))
            ->get();

        foreach ($overdueInvoices as $invoice) {
            // Send overdue reminder
            $this->info("Sent overdue reminder for invoice #{$invoice->id}");
        }
    }

    private function suspendOverdueServices()
    {
        $this->info('Suspending overdue services...');

        // Find services with unpaid invoices past due date + grace period (e.g., 7 days)
        $services = Service::where('status', 'active')
            ->whereHas('user.invoices', function($query) {
                $query->where('status', 'unpaid')
                    ->whereDate('due_date', '<', now()->subDays(7));
            })
            ->get();

        foreach ($services as $service) {
            $service->status = 'suspended';
            $service->save();
            
            $this->info("Suspended service #{$service->id} due to non-payment");
        }
    }

    private function terminateCancelledServices()
    {
        $this->info('Terminating cancelled services...');

        $services = Service::where('status', 'cancelled')
            ->whereDate('updated_at', '<', now()->subDays(30))
            ->get();

        foreach ($services as $service) {
            $service->status = 'terminated';
            $service->save();
            
            $this->info("Terminated service #{$service->id}");
        }
    }

    private function calculateNextDueDate($billingCycle, $currentDueDate)
    {
        $dueDate = Carbon::parse($currentDueDate);

        switch ($billingCycle) {
            case 'monthly':
                return $dueDate->addMonth();
            case 'quarterly':
                return $dueDate->addMonths(3);
            case 'semi_annually':
                return $dueDate->addMonths(6);
            case 'annually':
                return $dueDate->addYear();
            case 'biennially':
                return $dueDate->addYears(2);
            case 'triennially':
                return $dueDate->addYears(3);
            default:
                return $dueDate->addMonth();
        }
    }
}
