# MySQL Setup Guide for BillingCS

This guide provides detailed instructions for setting up BillingCS with MySQL for production use.

## Prerequisites

- MySQL 8.0+ or MariaDB 10.6+
- PHP 8.2+ with pdo_mysql extension enabled
- Composer and Node.js installed

## Step 1: Create MySQL Database

### Method 1: Using MySQL Command Line

```bash
# Login to MySQL
mysql -u root -p

# Create database with UTF-8 support
CREATE DATABASE billingcs CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

# Create a dedicated user (recommended for production)
CREATE USER 'billingcs_user'@'localhost' IDENTIFIED BY 'your_secure_password';

# Grant all privileges to the user
GRANT ALL PRIVILEGES ON billingcs.* TO 'billingcs_user'@'localhost';

# Flush privileges
FLUSH PRIVILEGES;

# Exit MySQL
EXIT;
```

### Method 2: Using phpMyAdmin

1. Login to phpMyAdmin
2. Click on "New" in the left sidebar
3. Enter database name: `billingcs`
4. Select "utf8mb4_unicode_ci" as collation
5. Click "Create"

### Method 3: Using cPanel

1. Login to cPanel
2. Navigate to "MySQL Databases"
3. Create a new database named "billingcs"
4. Create a new MySQL user
5. Add the user to the database with "All Privileges"

## Step 2: Configure Environment File

Edit your `.env` file:

```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=billingcs
DB_USERNAME=billingcs_user
DB_PASSWORD=your_secure_password

# For remote database servers
# DB_HOST=your-database-server.com
# DB_PORT=3306
```

## Step 3: Verify PHP MySQL Extension

Ensure the PDO MySQL extension is installed and enabled:

```bash
# Check if pdo_mysql is enabled
php -m | grep pdo_mysql

# If not installed (Ubuntu/Debian)
sudo apt-get install php8.2-mysql

# If not installed (CentOS/RHEL)
sudo yum install php82-mysqlnd

# Restart PHP-FPM or Apache
sudo systemctl restart php8.2-fpm
# or
sudo systemctl restart apache2
```

## Step 4: Run Migrations

```bash
# Clear any cached config
php artisan config:clear

# Run migrations
php artisan migrate

# Seed the database with sample data
php artisan db:seed
```

## Step 5: Verify Database Connection

```bash
# Test database connection
php artisan tinker

# In tinker, run:
DB::connection()->getPdo();

# Should show PDO object without errors
exit
```

## Production Considerations

### 1. Database Configuration Optimization

Add these settings to your MySQL configuration (`/etc/mysql/my.cnf` or `/etc/my.cnf`):

```ini
[mysqld]
# Character Set
character-set-server=utf8mb4
collation-server=utf8mb4_unicode_ci

# InnoDB Settings for Laravel
innodb_buffer_pool_size=256M
innodb_log_file_size=64M
innodb_flush_log_at_trx_commit=2
innodb_flush_method=O_DIRECT

# Connection Settings
max_connections=151
wait_timeout=28800
interactive_timeout=28800

# Query Cache (MySQL 5.7 and earlier)
# query_cache_type=1
# query_cache_size=32M
```

### 2. Database Backup Strategy

Set up regular backups using cron:

```bash
# Add to crontab (crontab -e)
0 2 * * * /usr/bin/mysqldump -u billingcs_user -p'your_password' billingcs > /backups/billingcs_$(date +\%Y\%m\%d).sql
```

Or use a backup script:

```bash
#!/bin/bash
# Save as /usr/local/bin/backup-billingcs.sh

DB_NAME="billingcs"
DB_USER="billingcs_user"
DB_PASS="your_password"
BACKUP_DIR="/var/backups/mysql"
DATE=$(date +"%Y%m%d_%H%M%S")

# Create backup directory if it doesn't exist
mkdir -p $BACKUP_DIR

# Create backup
mysqldump -u $DB_USER -p$DB_PASS $DB_NAME | gzip > $BACKUP_DIR/billingcs_$DATE.sql.gz

# Keep only last 7 days of backups
find $BACKUP_DIR -name "billingcs_*.sql.gz" -mtime +7 -delete

# Make executable
chmod +x /usr/local/bin/backup-billingcs.sh
```

### 3. Security Recommendations

1. **Use Strong Passwords**
   - Generate secure passwords for database users
   - Never use default passwords in production

2. **Limit Database User Privileges**
   ```sql
   -- Only grant necessary privileges
   GRANT SELECT, INSERT, UPDATE, DELETE, CREATE, DROP, INDEX, ALTER 
   ON billingcs.* TO 'billingcs_user'@'localhost';
   ```

3. **Firewall Configuration**
   - Only allow database connections from application server
   - Block external MySQL port (3306) if database is on same server

4. **SSL/TLS Connections**
   - Enable SSL for MySQL connections if database is on a different server
   - Update `.env`:
     ```env
     DB_SSL_CA=/path/to/ca-cert.pem
     DB_SSL_VERIFY_SERVER_CERT=true
     ```

### 4. Performance Tuning

1. **Enable Query Logging (Development Only)**
   ```env
   DB_LOG_QUERIES=true
   LOG_LEVEL=debug
   ```

2. **Use Database Indexing**
   - All foreign keys are automatically indexed
   - Consider adding indexes for frequently queried columns

3. **Connection Pooling**
   - Use persistent connections in production:
     ```php
     // config/database.php - mysql connection
     'options' => [
         PDO::ATTR_PERSISTENT => true,
     ],
     ```

## Troubleshooting

### Error: "SQLSTATE[HY000] [1045] Access denied"

**Solution**: Check your database credentials in `.env` file

### Error: "SQLSTATE[HY000] [2002] Connection refused"

**Solution**: 
1. Ensure MySQL service is running: `sudo systemctl status mysql`
2. Check if MySQL is listening on correct port: `netstat -tlnp | grep 3306`
3. Verify DB_HOST in `.env` file

### Error: "PDO MySQL driver not found"

**Solution**: Install PHP MySQL extension:
```bash
sudo apt-get install php8.2-mysql
sudo systemctl restart php8.2-fpm
```

### Error: "Specified key was too long"

**Solution**: This is a Laravel configuration issue with older MySQL versions.
The migrations are already configured correctly with `utf8mb4`, but if you encounter this:

```php
// In AppServiceProvider.php boot() method
use Illuminate\Support\Facades\Schema;
Schema::defaultStringLength(191);
```

### Slow Query Performance

**Solution**:
1. Enable MySQL slow query log
2. Analyze slow queries
3. Add appropriate indexes
4. Consider query caching (Redis)

## Migration from SQLite to MySQL

If you have existing data in SQLite and want to migrate:

1. **Export SQLite data**
   ```bash
   php artisan db:seed --class=ExportDataSeeder
   ```

2. **Switch to MySQL** (follow steps above)

3. **Import data**
   ```bash
   php artisan migrate:fresh --seed
   # Then manually import your data
   ```

## Testing Database Connection

Create a simple test route to verify database connectivity:

```php
// routes/web.php
Route::get('/test-db', function () {
    try {
        DB::connection()->getPdo();
        $tables = DB::select('SHOW TABLES');
        return response()->json([
            'status' => 'success',
            'message' => 'Database connected successfully',
            'tables' => count($tables),
            'database' => config('database.connections.mysql.database')
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'status' => 'error',
            'message' => $e->getMessage()
        ], 500);
    }
});
```

Visit `/test-db` to verify connection.

## Summary

You now have BillingCS configured with MySQL for production use. Key points:

✅ MySQL database created with proper character set
✅ Dedicated database user with appropriate privileges
✅ Environment configured for MySQL
✅ Migrations run successfully
✅ Database connection verified
✅ Production optimizations applied
✅ Backup strategy implemented

For additional support, refer to:
- [Laravel Database Documentation](https://laravel.com/docs/11.x/database)
- [MySQL 8.0 Reference Manual](https://dev.mysql.com/doc/refman/8.0/en/)
- [BillingCS Documentation](./README.md)
